<?php

namespace App\Http\Controllers\Admin;

use Symfony\Component\HttpFoundation\StreamedResponse;
use App\Http\Controllers\Controller;
use App\Models\Booking;
use Illuminate\Http\Request;

class SalesReportController extends Controller
{
    public function index(Request $request)
{
    $pageTitle = 'Sales Report';

    // Query base
    $query = Booking::with(['bookedRooms.room.roomType', 'payments']);

    // Date filter
    if ($request->date_from) {
        $query->whereDate('created_at', '>=', $request->date_from);
    }
    if ($request->date_to) {
        $query->whereDate('created_at', '<=', $request->date_to);
    }

    // Fetch data
    $bookings = $query->get()->map(function ($b) {
        $guest = is_string($b->guest_details)
            ? json_decode($b->guest_details)
            : (object) $b->guest_details;
        $guestName = $guest->name ?? ($guest->first_name ?? 'N/A');

        // Combine all booked room numbers
        $roomNumbers = $b->bookedRooms
            ->filter(fn($r) => $r->room)
            ->map(fn($r) => $r->room->room_number)
            ->implode(', ');

        // Combine all room types
        $roomTypes = $b->bookedRooms
            ->filter(fn($r) => $r->room && $r->room->roomType)
            ->map(fn($r) => $r->room->roomType->name)
            ->unique()
            ->implode(', ');

        $totalAmount = $b->booking_fare + $b->tax_charge + $b->service_cost + $b->extra_charge - $b->extra_charge_subtracted;
        $paid = $b->payments->sum('amount');

        return (object)[
            'id' => $b->id,
            'booking_number' => $b->booking_number,
            'guest_name' => $guestName,
            'checkin_date' => $b->check_in,
            'checkout_date' => $b->check_out,
            'total_amount' => $totalAmount,
            'total_paid' => $paid,
            'due' => $totalAmount - $paid,
            'status' => $this->statusLabel($b->status),
            'room_type' => $roomTypes ?: 'N/A',
            'room_no' => $roomNumbers ?: 'N/A',
            'created_at' => $b->created_at,
        ];
    });

    // Totals
    $totals = [
        'amount' => $bookings->sum('total_amount'),
        'paid'   => $bookings->sum('total_paid'),
        'due'    => $bookings->sum('due'),
    ];

    // Print versions
    if ($request->has('print')) {
        $view = $request->print === 'thermal'
            ? 'admin.reports.print_thermal'
            : 'admin.reports.print_a4';
        return view($view, compact('bookings', 'totals'));
    }

    return view('admin.reports.sales_report', compact('pageTitle', 'bookings', 'totals'));
}


    private function statusLabel($status)
    {
        switch ($status) {
            case 1:
                return 'Confirmed';
            case 3:
                return 'Cancelled';
            case 9:
                return 'Checked Out';
            default:
                return 'Pending';
        }
    }
    
    
    
    
public function exportExcel(Request $request)
{
    // We’ll reuse the same logic — export as CSV for now
    return $this->exportCsv($request);
}

public function exportCsv(Request $request)
{
    $filename = 'sales_report_' . now()->format('Y_m_d_H_i_s') . '.csv';
    $response = new StreamedResponse(function () use ($request) {
        $handle = fopen('php://output', 'w');
        fputcsv($handle, ['Booking No', 'Guest', 'Check In', 'Check Out', 'Total', 'Paid', 'Due', 'Status', 'Room Type', 'Room No', 'Date']);

        $bookings = Booking::with(['bookedRooms.room.roomType', 'payments'])
            ->when($request->date_from, fn($q) => $q->whereDate('created_at', '>=', $request->date_from))
            ->when($request->date_to, fn($q) => $q->whereDate('created_at', '<=', $request->date_to))
            ->get();

        foreach ($bookings as $b) {
            $guest = is_string($b->guest_details)
            ? json_decode($b->guest_details)
            : (object) $b->guest_details;
            $guestName = $guest->name ?? ($guest->first_name ?? 'N/A');
            $totalAmount = $b->booking_fare + $b->tax_charge + $b->service_cost + $b->extra_charge - $b->extra_charge_subtracted;
            $paid = $b->payments->sum('amount');
            $due = $totalAmount - $paid;

            fputcsv($handle, [
                $b->booking_number,
                $guestName,
                $b->check_in,
                $b->check_out,
                $totalAmount,
                $paid,
                $due,
                $this->statusLabel($b->status),
                optional($b->bookedRooms->first()->room->roomType)->name ?? 'N/A',
                optional($b->bookedRooms->first()->room)->room_number ?? 'N/A',
                $b->created_at,
            ]);
        }

        fclose($handle);
    });

    $response->headers->set('Content-Type', 'text/csv');
    $response->headers->set('Content-Disposition', "attachment; filename={$filename}");

    return $response;
}
    
    
    
    
    
    
}
